/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xsequenc.c 
 *
 * Generic sequence prepare functions.
 * Prepares the data settings for the sequencer so the sequence
 * compiler can read it. 
 * Programs hardware.                                 
 * -----------------------------------------------------------------*/

#include <string.h>
#include <stdio.h>
#include "xpciapi.h"
#include "xsequenc.h"
#include "xseqpars.h"
#include <xaddrmap.h>
#include <ctype.h>

#include "xutil.h"

#ifdef BEST_DEBUG
/* Always use it with two bracket pairs, e.g.
   DBG_OUT(("A=%lu B=%lu C=%lu\n",a,b,c));
   This makes the preprocessor handle it as only one 
   parameter (containing multiple parameters enclosed
   in brackets).
*/
#define DBG_OUT(a)          BESTX_PRINTF a
#else
#define DBG_OUT(a)          (void) (0)
#endif

/* Pointer to i'th sequencer in database */
#define DB_PERFSEQ(i) (bx_handlearray[handle].db->Analyzer.PerfSeq[i])
#define DB_TRIGSEQ(i) (bx_handlearray[handle].db->Analyzer.TrigSeq[i])

#ifndef MAX /*Defined in aix51B/com1/CMVC/export/power/usr/include/sys/ltypes.h */
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif

/* ------------------------------------------------------------------------- */
/* The format of the following data is as follows:                           */
/* - name of input or output (states are coded with an _sX_)                 */
/* - input number (the number for the MUX) or output signal position         */
/* - vectorlength                                                            */
/* - Property that this output is reflected by from C-API                    */
/* ------------------------------------------------------------------------- */

#define MUXOFFSET 0x4 /* difference between first MUX register and sequencer offset */
#define PLOADOFFSET  (BX_REG_BLK0_PERF_PL_REG - BX_REG_BLK0_RAM_CNTR) /* offset to prload register */
#define BX_NO_PROP -1

/* This struct holds information for the sequencer 
   input and output structures and some enums to 
   identify the conditions
*/

typedef struct 
{ 
  char * name;
  int position;
  int vectorlen;
  int enumid;
} seq_table_s;

typedef struct 
{
  const bx_seqtype sequencer;
  const seq_table_s   *seq_in;
  const seq_table_s   *seq_out;
  const int           inputlines;
  const bx_int32      seqoffset;
} seq_descr_table_s;



/* ---------------------------------------------------------------------- */
/* trigger sequencer                                                      */
/* ---------------------------------------------------------------------- */

/* We need this string multiple times (bugfix) */
#define BX_ERR0    "err0"
#define BX_NOTERR0 "!err0"

/* input data of the trigger sequencer */
static const seq_table_s trigger_seq_in[] = {
  { "_s0_",     0,       1,   BX_NO_PROP}, /* states                       */
  { "_s1_",     1,       1,   BX_NO_PROP},
  { "_s2_",     2,       1,   BX_NO_PROP},
  { "_s3_",     3,       1,   BX_NO_PROP},
  { "tc_fba",     4,       1,   BX_NO_PROP}, /* overflow feedback counter A  */
  { "tc_fbb",     5,       1,   BX_NO_PROP}, /* overflow feedback counter B  */
  { "bus0",    16,       1,   BX_NO_PROP}, /* bus patterns (97 bit)        */
  { "bus1",    17,       1,   BX_NO_PROP},
  { "bus2",    24,       1,   BX_NO_PROP},
  { "bus3",    25,       1,   BX_NO_PROP},
  { "obs0",    18,       1,   BX_NO_PROP}, /* observer patterns (101 bit)  */
  { "obs1",    19,       1,   BX_NO_PROP},
  { "obs2",    20,       1,   BX_NO_PROP},
  { "obs3",    21,       1,   BX_NO_PROP},
  { "obs4",    26,       1,   BX_NO_PROP},
  { "obs5",    27,       1,   BX_NO_PROP},
  { "tran0",    22,       1,   BX_NO_PROP}, /* transitional pattern (96 bit)*/
  { BX_ERR0,    23,       1,   BX_NO_PROP}, /* error pattern (3 bit)        */
  { "gnd",    28,       1,   BX_NO_PROP}, /* Ground                       */
  { NULL,     0,       0,   BX_NO_PROP}
};

/* output data of the trigger sequencer 
   CAUTION: BX_TRIGCOND_X does not appear as output ! */
static const seq_table_s trigger_seq_out[] = {
  { "_s0_",     0,       1,   BX_NO_PROP}, 
  { "_s1_",     1,       1,   BX_NO_PROP},
  { "_s2_",     2,       1,   BX_NO_PROP},
  { "_s3_",     3,       1,   BX_NO_PROP},
  { "dec_fba",     4,       1,   BX_TRIGCOND_FBADEC},
  { "ld_fba",     5,       1,   BX_TRIGCOND_FBALOAD},
  { "dec_fbb",     6,       1,   BX_TRIGCOND_FBBDEC}, 
  { "ld_fbb",     7,       1,   BX_TRIGCOND_FBBLOAD}, 
  { "sq",     8,       1,   BX_TRIGCOND_SQ},
  { "trig",     9,       1,   BX_TRIGCOND_TRIG},
  { NULL,     0,       0,   BX_NO_PROP}
};


/* ---------------------------------------------------------------------- */
/* performance sequencer                                                  */
/* ---------------------------------------------------------------------- */

/* input data of the performance sequencer */
static const seq_table_s performance_seq_in[] = {
  { "_s0_",     0,       1,   BX_NO_PROP}, /* states                       */
  { "_s1_",     1,       1,   BX_NO_PROP},
  { "_s2_",     2,       1,   BX_NO_PROP},
  { "_s3_",     3,       1,   BX_NO_PROP},
  { "tc_fba",     4,       1,   BX_NO_PROP}, /* overflow feedback counter A  */
/*{ "tc_fbb",     5,       1,   BX_NO_PROP}, no feedback counter B here !!   */
  { "gnd",     5,       1,   BX_NO_PROP}, /* Ground                       */
  { "bus0",    16,       1,   BX_NO_PROP}, /* bus patterns (97 bit)        */
  { "bus1",    17,       1,   BX_NO_PROP},
  { "bus2",    24,       1,   BX_NO_PROP},
  { "bus3",    25,       1,   BX_NO_PROP},
  { "obs0",    18,       1,   BX_NO_PROP}, /* observer patterns (101 bit)  */
  { "obs1",    19,       1,   BX_NO_PROP},
  { "obs2",    20,       1,   BX_NO_PROP},
  { "obs3",    21,       1,   BX_NO_PROP},
  { "obs4",    26,       1,   BX_NO_PROP},
  { "obs5",    27,       1,   BX_NO_PROP},
  { "tran0",    22,       1,   BX_NO_PROP}, /* transitional pattern (96 bit)*/
  { BX_ERR0,    23,       1,   BX_NO_PROP}, /* error pattern (3 bit)        */
  { NULL,     0,       0,   BX_NO_PROP}
};

/* output data of the performance sequencer.
   CAUTION: BX_PERFCOND_X does not appear as output ! */
static const seq_table_s performance_seq_out[] = {
  { "_s0_",     0,       1,   BX_NO_PROP}, 
  { "_s1_",     1,       1,   BX_NO_PROP},
  { "_s2_",     2,       1,   BX_NO_PROP},
  { "_s3_",     3,       1,   BX_NO_PROP},
  { "dec_fba",     4,       1,   BX_PERFCOND_FBADEC},  /* Feedback ctr  */
  { "ld_fba",     5,       1,   BX_PERFCOND_FBALOAD},
  { "inc_ctra",     6,       1,   BX_PERFCOND_CTRAINC}, /* Byte counter */
  { "inc_ctrb",     7,       1,   BX_PERFCOND_CTRBINC}, /* Incrementor  */
/*{ "sq",     8,       1,   BX_PERFCOND_SQ}, no sq in HW here !!! */
  { "crule",     9,       1,   BX_NO_PROP}, 
  { NULL,     0,       0,   BX_NO_PROP}
};


/* ---------------------------------------------------------------------- */
/* table with all sequencers available                                    */
/* ---------------------------------------------------------------------- */

static const seq_descr_table_s seq_descr_table[] = 
{
  { BX_SEQ_TRIG,trigger_seq_in,trigger_seq_out,7,BX_REG_TRIG_SEQ_MEM},

  { BX_SEQ_PERF_0,performance_seq_in,performance_seq_out,7,BX_REG_BLK0_RAM_CNTR + BX_PERFSEQ_ADDR_OFFS * 0},

  { BX_SEQ_PERF_1,performance_seq_in,performance_seq_out,7,BX_REG_BLK0_RAM_CNTR + BX_PERFSEQ_ADDR_OFFS * 1},

  { BX_SEQ_PERF_2,performance_seq_in,performance_seq_out,7,BX_REG_BLK0_RAM_CNTR + BX_PERFSEQ_ADDR_OFFS * 2},

  { BX_SEQ_PERF_3,performance_seq_in,performance_seq_out,7,BX_REG_BLK0_RAM_CNTR + BX_PERFSEQ_ADDR_OFFS * 3},

  { BX_SEQ_INVALID, NULL, NULL, 0, 0} 
};

/******************************************************************************
 **** Sequencer programming functions
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * ListFinder
 *
 * Purpose: searches a list for a specific token 
 *---------------------------------------------------------------------------*/

static int ListFinder(const seq_table_s * list,const char * txt)
{
  int k = 0;
  assert(txt);
  while(list[k].name && 
    (BESTX_STRCMP(list[k].name,txt) != 0))
    k++; 
  assert(list[k].name); /* make sure it is really there!!! */
  return k;  
}


/*---------------------------------------------------------------------------*
 * BestXSequencerProg()
 *
 * Purpose: Sequencer programming.
 *          Programs any sequencer and parses all interesting equations 
 *---------------------------------------------------------------------------*/

bx_errtype BestXSequencerProg (
  bx_handletype handle,
  bx_seqtype sequencer   /* what sequencer to take*/
)            
{
  size_t      i,j;
  bx_int32         k;         /* used for for loops all over the function       */
  int         gndindex;           /* index to identify the gnd line                 */
  char *      bp,*cp;             /* points to the buffer (for scanning purposed)   */
  static char buf[YYSEQBUFSIZE];  /* holds one line of the trigger sequencer        */
  bx_int32         num_states;    /* the number of states we need                   */
  bx_int32         max_ldstates;  /* the maximum number of states we can use        */
  bx_int32         max_ldoutputs; /* the maximum number of outputlines we can use   */
  bx_int32         used_ldoutputs;/* the number of used outputs                     */
  bx_int32         max_ldinputs;  /* the maximum number of inputlines we can use    */
  bx_int32         ldstates,statedp; /* logarithm of num_states (base 2) (twice)    */
  bx_int32         input_linecount; /* count the number of input lines              */
  bx_int32         output_linecount; /* count the number of output lines            */
  token       *state_tokenlist; /* the state tokens                          */
  token       *output_tokenlist; /* the output tokens                        */
  token       *input_tokenlist; /* the input tokens                          */
  sequencer_descriptor seqdescript; /* the sequence descriptor               */
  bx_int32    ramline_lo,ramline_hi;/* ram memory line (high and low value)  */
  bx_int32    seqAddr;          /* the sequencer start address               */
  bx_int32    mask6;            
  bx_int32    mask7;
  bx_int32    val;
  bx_int32    mephisto_version;

  int    strposofxcond; /* the position of the xcond in the first line  */

  char        conditionUse[MAX(BX_PERFLINECOND_SIZE,BX_TRIGLINECOND_SIZE)+1]; 
               /* holds the hashtable for conditions to use */
               /* The "+1" is needed for the while(conditionUse[j])-loop, which relies
                  on the enumid=0 as termination condition in the sequencer table
                */

  char        token[32];        /* token identifier                          */
  int         inputarray[64];   /* remember the input tokens we need         */

  const seq_table_s *seqInputs; /* this is the sequencer inputs table        */
  const seq_table_s *seqOutputs;/* this is the sequencer output table        */

  bx_int32 max; /* number of lines to program (1-based) */
  bx_dbtriglinetype *mytrigseqline; 
  bx_dbperflinetype *myperfseqline;

  /* the size of this buffer is 4096 bytes - this might not be enough!!      */
  extern char yy_sequence_buffer[YYSEQBUFSIZE];  /* used by the parser       */

  BX_TRY_VARS; /* variables for try blocks */

  BX_TRY_BEGIN
  {
    /* get hold of mephisto version for bytecount bugfix ^2 */
    BX_TRY(BestXDirectRegRead(handle, BX_REG_CHIP_REV_REG, 2, &mephisto_version));

    /* Need to differenciate between performance- and trigger sequencers */
    if (sequencer==BX_SEQ_TRIG)
    {
      /* sequencer is a trigger sequencer */
      assert(sequencer-BX_SEQ_TRIG<BX_TRIGSEQ_SIZE);
      
      mytrigseqline=DB_TRIGSEQ(sequencer-BX_SEQ_TRIG).Line;
      max=DB_TRIGSEQ(sequencer-BX_SEQ_TRIG).MaxNumLines;

      assert(max<=BX_TRIGLINE_SIZE);
      myperfseqline=NULL;
    }        
    else    
    {
      /* sequencer is a performance sequencer (guaranteed by caller) */
      /* do mephisto version checking as of bytcount bugfix ^2 */
      if(mephisto_version == BX_MEPHISTO_MMC1_MMC2)
      { 
        assert(BX_SEQ_PERF_0<=sequencer && sequencer<BX_SEQ_PERF_0+BX_PERFSEQ_SIZE_MMC1_MMC2 
          && "parameter sequencer is out of range");
        assert(sequencer-BX_SEQ_PERF_0<BX_PERFSEQ_SIZE_MMC1_MMC2);
      }
      else /*/ also used in OFFLINE case as mephisto_version is 0x0 */
      {
        assert(BX_SEQ_PERF_0<=sequencer && sequencer<BX_SEQ_PERF_0+BX_PERFSEQ_SIZE 
          && "parameter sequencer is out of range");
        assert(sequencer-BX_SEQ_PERF_0<BX_PERFSEQ_SIZE);
      }
      myperfseqline=DB_PERFSEQ(sequencer-BX_SEQ_PERF_0).Line;
      max=DB_PERFSEQ(sequencer-BX_SEQ_PERF_0).MaxNumLines;
      assert(max<=BX_PERFLINE_SIZE);

      mytrigseqline=NULL;
    }
 

    if(max == 0) /* no line to compile */
    {
      /* switch the sequencer off ?? */

      DBG_OUT(("  No line to compile - nothing to do.\n"));

      BX_ERRETURN(BX_TRY_RET); /* nothing to do here */
    }

    /* start of computation */

    DBG_OUT (("Sequencer Table:\n"));

    /* initialize the yy buffer */
    /* the yy buffer has a size of 4096 bytes - maybe allocate it dynamically? */
    yy_sequence_buffer[0] = '\0';

    /* initialize all other variables */
    num_states = 0;
    for(i=0;i<64;i++)
    {
      inputarray[i] = 0;
    }
    state_tokenlist = NULL; 
    output_tokenlist = NULL;
    input_tokenlist = NULL;


    /* --------------------------------------------------------------*/
    /* first find the sequencer we are working with and get the info */
    /* --------------------------------------------------------------*/
    i = 0;
    seqInputs = NULL;
    seqOutputs = NULL;
    while(seq_descr_table[i].sequencer != BX_SEQ_INVALID)
    {
      if(seq_descr_table[i].sequencer == sequencer) /* sequencer found */
      {
        seqInputs  = seq_descr_table[i].seq_in;
        seqOutputs = seq_descr_table[i].seq_out;
        seqAddr    = seq_descr_table[i].seqoffset;
        break;
      }
      i++;
    }
    if(seqInputs == NULL) /* we did not find the sequencer specified */
    {
      DBG_OUT(("Sequencer '%d' not found.\n",i));
      BX_ERRETURN(BX_E_PARAM);
    }


    /* -------------------------------------------------------------*/
    /* create a hash table to read output properties that are needed*/
    /* -------------------------------------------------------------*/
    for(i=0;i<MAX(BX_PERFLINECOND_SIZE,BX_TRIGLINECOND_SIZE)+1;i++)
    {
      /* The "+1" is important here, because it sets the termination condition
         for the while(conditionUse[j])-loop !! */
      conditionUse[i] = '\0'; /* mark them invalid -> the condition use field */
    }

    i = 0;
    j = 0;
    while(seqOutputs[i].name) /* run through all output signals */
    {
      if(seqOutputs[i].enumid != BX_NO_PROP)
      {
        /* CAPI property found (exception: BX_TRIGCOND_X (resp. BX_PERFCOND_X) does
           not appear in output table !) */
        
        /* Remember the index into output table */
        conditionUse[j++] = (char) i;/* i is always bigger than zero, so this is legal */
      }
      i++;
    }

    /* The only condition property not covered by the output table is the 
       COND_X property. We use 255 as a special,invalid index value into
       the output table here.
    */
    conditionUse[j++] = (char) 255; /* this is the trick to access xcond too */

    /* j+1 is the lowest "invalid" index for array conditionUse[] here */
    assert(j==(bx_int32)(sequencer==BX_SEQ_TRIG?BX_TRIGLINECOND_SIZE:BX_PERFLINECOND_SIZE));

    /* -------------------------------------------------------------*/
    /* loop over all lines. The following happens here:
       - printing of the complete string for the parser 
       - check for state numbers (maximum)
       - check for needed input terms (pattern and terminal counts)
       - determine the position of the xcond string for the first line
         ( needed in case there is only one line and we have no condition
           to work with. In this case the compiler needs a dummy condition
       to avoid a crash).
       */
    
    
    /* -------------------------------------------------------------*/
    strposofxcond = -1; /* the default position of the xcond in the first line */
    for(i=0;i<max;i++)  
    {
      /* using the hash table from a few lines above the output strings get
       * concatenated here. They are printed line by line and all lines are
       * concatenated in the yy_sequence_buffer */

      /* first print the start of the line ... */
      if (sequencer==BX_SEQ_TRIG)
      {
        sprintf(buf," T%d: state=%d nextstate=%d xcond=<%s> ",i,
            mytrigseqline[i].Transient[BX_TRIGTRAN_STATE],
            mytrigseqline[i].Transient[BX_TRIGTRAN_NEXTSTATE],
            (mytrigseqline[i].Condition[BX_TRIGCOND_X] ? mytrigseqline[i].Condition[BX_TRIGCOND_X]:"0"));
      }
      else
      {
        sprintf(buf," T%d: state=%d nextstate=%d xcond=<%s> ",i,
            myperfseqline[i].Transient[BX_PERFTRAN_STATE],
            myperfseqline[i].Transient[BX_PERFTRAN_NEXTSTATE],
            (myperfseqline[i].Condition[BX_PERFCOND_X] ? myperfseqline[i].Condition[BX_PERFCOND_X]:"0"));
      }

      if(i==0)
      {
        strposofxcond = (int) (BESTX_STRSTR(buf,"xcond")-buf) + 7; /* 7==BESTX_STRLEN("xcond=<"); */
        /* strpsofxcond points now to the first xcond in the first line */
      }


      DBG_OUT((buf));  /* output the line for debugging */
      DBG_OUT(("\n"));  /* output the line for debugging */
      BESTX_STRCAT(yy_sequence_buffer,buf);

      /* no output conditions are set so far... */

      /* catch all states we are using - the highest number is the max states */
      /* we need this information to create the information structures for 
         the trigger sequence parser */

      if (sequencer==BX_SEQ_TRIG)
      {
        if(num_states <= mytrigseqline[i].Transient[BX_TRIGTRAN_STATE])
        {
          num_states = mytrigseqline[i].Transient[BX_TRIGTRAN_STATE]+1;
        }

        if((bx_int32)num_states <= mytrigseqline[i].Transient[BX_TRIGTRAN_NEXTSTATE])
        {
          num_states = mytrigseqline[i].Transient[BX_TRIGTRAN_NEXTSTATE]+1;
        }
      }
      else
      {
        if(num_states <= myperfseqline[i].Transient[BX_PERFTRAN_STATE])
        {
          num_states = myperfseqline[i].Transient[BX_PERFTRAN_STATE]+1;
        }

        if(num_states <= myperfseqline[i].Transient[BX_PERFTRAN_NEXTSTATE])
        {
          num_states = myperfseqline[i].Transient[BX_PERFTRAN_NEXTSTATE]+1;
        }
      }

      /* num_states contains now the maximum state used up to now plus one */ 

      /* find and identify the input tokens we need */
      j = 0;
      while(conditionUse[j]) /* j counts condition props; conditionUse[j] is corresponding index into output table */
      {
        /* access the property (use the hash table from above) */
        if(conditionUse[j] == (char) 255)
        {
          /* marks last entry in array conditionUse[], which represents COND_X condition property */
          if (sequencer==BX_SEQ_TRIG)
          {
            bp = mytrigseqline[i].Condition[BX_TRIGCOND_X];
          }
          else
          {
            bp = myperfseqline[i].Condition[BX_PERFCOND_X];
          }
        }
        else
        {
          if (sequencer==BX_SEQ_TRIG)
          {
            bp = mytrigseqline[i].Condition[seqOutputs[conditionUse[j]].enumid];
          }
          else
          {
            bp = myperfseqline[i].Condition[seqOutputs[conditionUse[j]].enumid];
          }
        }

        /* bp points to the j'th condition property string */
        
        cp = bp; /* we need the condition later on */

        if(bp) /* do this only if we have a valid pointer to a condition*/
        {
          /* Start parsing condition properties */

      while(*bp) /* step over the string */
      {
        token[0] = '\0';
        k = 0;
        /* token collection */
        while(*bp) /* collect the complete token (as far as possible) */
        {
              if (isalnum(*bp)||(*bp == '_')) /* [A-Za-z0-9_] */
              {
                token[k++] = *bp;
          }
              else
          {
                if(token[0]) /* break only if the token has at least one char */
            {
          bp++;
          break;
            }
              }
          bp++;
        }
        token[k++] = '\0';

            /* we have now in 'token' the tokenstring (as far as possible) */

            if(token[0] == '\0') /* did not find anything */
        {
              break; /* leave the while loop and look for the next cond string */
            }

        /* search the input token list for the token */
        k=0;
        input_linecount = 0;
        while(seqInputs[k].name)
        {
          if((seqInputs[k].name[0] != '_') && 
             ((BESTX_STRCMP(seqInputs[k].name,token) == 0) || 
          (BESTX_STRCMP(token,"1") == 0) || 
          (BESTX_STRCMP(token,"0") == 0)))
          { /* found token ... */
            /* do not do this if we work with xcondition */
            if(conditionUse[j] != (char) 255)
            {
          /* create the next cond string in the line for the parser */
          sprintf(buf,"%s=<%s> ",seqOutputs[conditionUse[j]].name,cp);
          BESTX_STRCAT(yy_sequence_buffer,buf);
            }

            /* mark it in the list of inputs... (array to be made up front)*/
            if (!((BESTX_STRCMP(token,"1") == 0) || 
              (BESTX_STRCMP(token,"0") == 0)))
          /*inputarray[seqInputs[k].position] = 1; WRONG!!! */
          inputarray[k] = 1; /* set a bit for every input used */
            break;
            /* we assume one bit signals so far as inputs !!!! */
          }
          k++;
        }

            /* we either found the token or not */
        if(seqInputs[k].name == NULL) /* input token not found!! */
        { 
          /* call the error routine to set all error parameters */
          BestXLastErrorParamSet(handle,BX_ERRPAR_1,sequencer);
          BestXLastErrorParamSet(handle,BX_ERRPAR_2,(bx_int32)i);
          BestXLastErrorParamSet(handle,BX_ERRPAR_3,0);
          BX_ERRETURN (BX_E_SEQ_UNKNOWN_TOKEN); 
        }

      } /* end of while (*bp) loop */

        } /* end of if(cond) was valid at all */

        j++; /* increase j to advance in the while loop */
      } /* end of running through all inputs */

      /* add the end-line sign to the buffer */
      BESTX_STRCAT(yy_sequence_buffer,";");
    } /* end of loop through all lines */


    /* finish and mark the end of the sequence description table */
    yy_sequence_buffer[BESTX_STRLEN(yy_sequence_buffer) - 1] = '.';

    DBG_OUT((" Sequencer Prg: Buffer: %d Bytes, Max: %d Bytes, Buffer: <%s>\n", 
        BESTX_STRLEN(yy_sequence_buffer),YYSEQBUFSIZE,yy_sequence_buffer));
    

    /* count the number of states we need */

    DBG_OUT((" Sequencer Prg: States: %d\n",num_states));

    /* -------------------------------------------------------------*/
    /* ----------------- state token list --------------------------*/
    /* -------------------------------------------------------------*/
    k = 0;
    max_ldstates = 0;
    while(seqInputs[k].name) /* count the maximum number of state inputs */
    {
      if(seqInputs[k].name[0] == '_')
      {
        max_ldstates++;
      }
      k++;
    }

    /* calculate the logarithm of the number of states (base 2) */
    statedp = num_states;

    /* new way of calculating the needed cables */
    for(ldstates = 0; (bx_int32)(1<<ldstates) < statedp; ldstates++)
    {
      ;
    }

    /* and check if the maximum number of allowed lines is exceeded */
    if(ldstates > max_ldstates)
    {
      /* add some information here too... */
      
      BestXLastErrorParamSet(handle,BX_ERRPAR_1,sequencer);
      BestXLastErrorParamSet(handle,BX_ERRPAR_2,(bx_int32)i);
      
      BX_TRY_FAIL (BX_E_SEQ_TOO_MANY_STATES); 
    }
    

    
    if(ldstates>0)
    {
      /* create a new list of states according to the state information given above */
      state_tokenlist=BestXMemMalloc((size_t)(ldstates * sizeof(token)));
      BX_TRY_FAIL(state_tokenlist!=NULL ? BX_E_OK : BX_E_HOST_MEM_FULL);
    }
    else
    {
      state_tokenlist=NULL;
    }

    for(i=0;i<ldstates;i++)
    {
      state_tokenlist[i].tok_name = NULL;
    }

    for(i=0;i<ldstates;i++)
    {
      state_tokenlist[i].tok_name=BestXMemMalloc(8);
      BX_TRY_FAIL(state_tokenlist[i].tok_name?BX_E_OK:BX_E_HOST_MEM_FULL);

      sprintf(state_tokenlist[i].tok_name,"_s%d_",i);
      state_tokenlist[i].vectorlen = (bx_int32)i; /* input index */
      state_tokenlist[i].position = (bx_int32)i;  /* output index */
    }

    /* -------------------------------------------------------------*/
    /* ----------------- output token list -------------------------*/
    /* -------------------------------------------------------------*/
    /* create the list of outputs (including a dummy output for the not
       needed states) */

    used_ldoutputs = 0;
    /* count the maximum number of state outputs */
    max_ldoutputs = 0;
    while(seqOutputs[max_ldoutputs].name) 
    {
      max_ldoutputs++;
    }

    output_tokenlist=BestXMemMalloc((size_t)(max_ldoutputs * sizeof(token)));
    BX_TRY_FAIL(output_tokenlist?BX_E_OK:BX_E_HOST_MEM_FULL);

    /* zero all name pointer - just in case we do a delete with it*/
    for(i=0;i<max_ldoutputs;i++)
    {
      output_tokenlist[i].tok_name = NULL;
    }

    /* run through all outputs and add them to the list */
    k = 0;
    output_linecount = 0;
    while(seqOutputs[k].name) /* add all outputs from the table above */
    {
      if(seqOutputs[k].name[0] != '_')
      {
        output_tokenlist[output_linecount].tok_name = seqOutputs[k].name;
        output_tokenlist[output_linecount].vectorlen = seqOutputs[k].vectorlen;
        output_tokenlist[output_linecount].position  = seqOutputs[k].position;
        output_linecount += seqOutputs[k].vectorlen;
        used_ldoutputs++;
      }
      k++;     
    }

    /* and here follows the dynamic part depending on how many 
       state outputs we do not need and can use for dummies */
    /* we use another n dummy outputs. The number n calculates as
       the maximum number of possble state outputs minus the number
       of used state outputs (remains the number of not used state outputs) */
    for(i=0;i<(max_ldstates - ldstates);i++,output_linecount++)
    {
      output_tokenlist[output_linecount].tok_name=BestXMemMalloc(10);

      BX_TRY_FAIL(output_tokenlist[output_linecount].tok_name?BX_E_OK:BX_E_HOST_MEM_FULL);

      sprintf(output_tokenlist[output_linecount].tok_name,"_out%d_",
          used_ldoutputs+i);
      output_tokenlist[output_linecount].vectorlen = 1;
      output_tokenlist[output_linecount].position = ldstates+(bx_int32)i;
    }

    /* -------------------------------------------------------------*/
    /* ----------------- input token list --------------------------*/
    /* -------------------------------------------------------------*/

    /* count the maximum number of input lines */
    /* do NOT use the list - obtain it from the information array */
    /* with the mux there are way to many possible input signals */
    max_ldinputs = seq_descr_table->inputlines;

    /* max_ldinputs is a little to high, depends on states but who cares */

    input_tokenlist=BestXMemMalloc((size_t)((max_ldinputs + 2) * sizeof(token)));
    BX_TRY_FAIL(input_tokenlist?BX_E_OK:BX_E_HOST_MEM_FULL);

    /* zero all pointer - just in case */
    for(i=0;i<max_ldinputs+2;i++)
    {
      input_tokenlist[i].tok_name = NULL;
    }

    input_linecount = 0; /* count how many inputs we have... */
    for(i=0;i<64;i++) /* look for pattern terms... */
    {
      if(inputarray[i])
      {
        if(input_linecount > (max_ldinputs - ldstates))
        {
          /* add the values we need to the error reporting mechanism */
      /*FW_TRACERET (BX_E_SEQ_TOO_MANY_INPUTS);*/
      BestXLastErrorParamSet(handle,BX_ERRPAR_1,sequencer);
      BestXLastErrorParamSet(handle,BX_ERRPAR_2,max_ldinputs - ldstates);
      BestXLastErrorParamSet(handle,BX_ERRPAR_3,ldstates);
      BX_TRY_FAIL(BX_E_SEQ_TOO_MANY_INPUTS);
        }

        input_tokenlist[input_linecount].tok_name = seqInputs[i].name;
        input_tokenlist[input_linecount].vectorlen = seqInputs[i].vectorlen;
        /* start counting the inputs from zero always - these are NO real
           numbers, the input states are the ones that have priority and the
       input signals get numbered around them */
        input_tokenlist[input_linecount].position  = input_linecount;
        input_linecount++;
      }
    }

    /* in case we have no input lines, we need to create a dummy input
       (without actually using it in the mux) so the sequence compiler has
       at least one input to play with. The sequence compiler does not
       survive a zero input sequence. The dummy input we use is called 'Y'
       and gets a fictiv bit position of 0. It replaces whatever was in the
       xcond field (can only be a '0' or a '1'). We use the term 'Y | !Y' to
       replace a '1' in this field. And we use the term 'Y & !Y' to replace
       a '0' there. This should solve all possible situations that could 
       happen with a no input sequencer with only one state. */
    if((input_linecount == 0) && (ldstates == 0) && (strposofxcond > -1))
    {
      
      DBG_OUT(("  Sequencer has no Input: Adding dummy input 'Y' \n"));
      
      /* first add a dummy token */
      input_tokenlist[input_linecount].tok_name = "Y";
      input_tokenlist[input_linecount].vectorlen = 1;
      input_tokenlist[input_linecount].position  = 0;
      input_linecount++;
      /* and now insert the replacement string into the first xcond string */ 
      /* the position is given by strposofxcond */
      /* remember what is was */
      if( yy_sequence_buffer[strposofxcond] == '0')
      {
        cp = "Y & !Y";
      }
      else
      {
        cp = "Y | !Y";
      }
      /* zero the string at this position */
      yy_sequence_buffer[strposofxcond] = '\0';
      /* copy the sequence back to buf */
      BESTX_STRCPY(buf,yy_sequence_buffer);
      BESTX_STRCAT(buf,cp);
      BESTX_STRCAT(buf,yy_sequence_buffer + strposofxcond + 1);
      BESTX_STRCPY(yy_sequence_buffer,buf);

      
      DBG_OUT(("New buffer looks like this: <%s>\n",
          yy_sequence_buffer)); /* output the line for debugging */
    }
    else
    {
      strposofxcond = -1; /* mark the replacement as invalid so we have an
      indication if we added a dummy parameter for the mux programming */
    }


#define ERR_BUGFIX
#ifdef ERR_BUGFIX
    /* HW-bug: We cannot OR the error pattern term, only ANDing possible.
       Solution: De Morgan helps us out:
       Replace "err0" by "!err0" here.
    */

    {
      size_t len,len2;

      buf[0]='\0';
      len=BESTX_STRLEN(yy_sequence_buffer);
      len2=BESTX_STRLEN(BX_ERR0);
      i=j=0;
      while (i<len)
      {
        if(BESTX_STRNCMP(yy_sequence_buffer+i,BX_ERR0,len2)==0)
        {
          BESTX_STRCAT(buf+j,BX_NOTERR0); /* "!err0" */
          i+=len2;
          j+=len2+1;
        }
        else
        {
          buf[j]=yy_sequence_buffer[i];
          buf[j+1]='\0';
          i++;
          j++;
        }
      }
    
      BESTX_STRCPY(yy_sequence_buffer,buf);

      DBG_OUT(("Very new buffer looks like this: <%s>\n",
      yy_sequence_buffer)); /* output the line for debugging */

    }
#endif

    /* now add '0' and '1' as pseudo signals */
    input_tokenlist[input_linecount].tok_name = "0";
    input_tokenlist[input_linecount].vectorlen = 1;
    input_tokenlist[input_linecount].position  = 254;
    input_linecount++;
    input_tokenlist[input_linecount].tok_name = "1";
    input_tokenlist[input_linecount].vectorlen = 1;
    input_tokenlist[input_linecount].position  = 255;
    input_linecount++;


    /* -------------------------------------------------------------*/
    /* ----------------- programming -------------------------------*/
    /* -------------------------------------------------------------*/

    /* some debugging output is generated now */
#ifdef BEST_DEBUG
      DBG_OUT((" Sequencer Prg: States Lines: %d\n",ldstates));
      for(i=0;i<ldstates;i++)  
      {
        DBG_OUT(("  StateLine: %2d, Name: \"%s\" \tVectorLen: %d Position: %d\n",i,
            state_tokenlist[i].tok_name,
            state_tokenlist[i].vectorlen,
            state_tokenlist[i].position));
      }
      DBG_OUT((" Sequencer Prg: Input  Lines: %d\n",input_linecount));
      for(i=0;i<input_linecount;i++)  
      {
        DBG_OUT(("  InputLine: %2d, Name: \"%s\" \tVectorLen: %d Position: %d\n",i,
            input_tokenlist[i].tok_name,
            input_tokenlist[i].vectorlen,
            input_tokenlist[i].position));
      }
      DBG_OUT((" Sequencer Prg: Output Lines: %d\n",output_linecount));
      for(i=0;i<output_linecount;i++)  
      {
        DBG_OUT(("  OutputLine: %2d, Name: \"%s\" \tVectorLen: %d Position: %d\n",i,
            output_tokenlist[i].tok_name,
            output_tokenlist[i].vectorlen,
            output_tokenlist[i].position));
      }
    
#endif

    /* initialize the sequencer parser */
    seqdescript.name = NULL;
    seqdescript.states = 1 << ldstates;
    seqdescript.inputs = input_linecount + ldstates - 2;
    seqdescript.outputs = max_ldoutputs;
    seqdescript.input_tokens = input_linecount - 2; /* '0' and '1' are missing */
    seqdescript.output_tokens = output_linecount;
    seqdescript.input_tokenlist = input_tokenlist;
    seqdescript.output_tokenlist = output_tokenlist;
    seqdescript.state_tokenlist = state_tokenlist;

    BX_TRY(InitSequencerDescriptor("Sequencer",&seqdescript)); 

    /* parse the sequence description */
    BestXLastErrorParamSet(handle,BX_ERRPAR_1,sequencer);
    BX_TRY(ParseSequenceDescription()); 

    k = 1 << seqdescript.inputs;
    
    DBG_OUT(("  Sequencer Result: %d Lines\n",k));
    

    /* ---------------------------------------------- */
    /* prepare for programming the sequencer register */

    /* switch the trigger sequencer to programming mode */
    if (sequencer==BX_SEQ_TRIG)
    {
      BX_TRY_PROGRESS(BestXAnalyzerProgMode(handle,0 /* no soft force */));
    }
    else
    {
      BX_TRY_PROGRESS(BestXPerfProgMode(handle,0 /* no soft force */));
    }

    /* the next section is depending on the amount of output lines the sequencer
       has and the resulting width of the sequencer RAM */
    /* first write the address 0 to the preload register.
       Its the same preload register for all performance sequencers */

    BX_TRY(BestXDirectRegWrite(handle,(sequencer==BX_SEQ_TRIG ?BX_REG_TRIG_RAM_PL_REG:BX_REG_PERF_RAM_PL_REG),sizeof(bx_int16),0x0));

    /* then get the strobe once */
    BX_TRY(BestXDirectRegWrite(handle,(sequencer==BX_SEQ_TRIG?BX_REG_TRIG_RAM_ADDR_PL_STRB:BX_REG_PERF_RAM_ADDR_PL_STRB),sizeof(bx_int16),0x0000));

    DBG_OUT(("  Sequencer Result: Now programming the Ram\n"));

    for(i=0;i<k;i++)
    {
      BX_TRY(ReadSequencerRamline((bx_int32)i,&ramline_lo,&ramline_hi)); 

      DBG_OUT(("  RamLine_lo: %03x, Value: %08x \n",i, ramline_lo)); 
      /* DBG_OUT("  RamLine_hi: %03x, Value: %08x: \n",i, ramline_hi); */ 
      /* program the ram line ... */

      BX_TRY(BestXDirectRegWrite(handle,seqAddr,sizeof(bx_int16),ramline_lo));
    }

    /* start programming the MUX that defines the input lines */

    DBG_OUT(("  Sequencer Result: Now programming Mux 0 to %d \n",
        max_ldinputs-1));
    

    /* first find "gnd" as a pulldown pattern */
    gndindex = ListFinder(seqInputs,"gnd");

    for(i=0;i<max_ldinputs;i++)
    {
      if(i < ldstates) /* one of the state lines (start with stateline 0 */
      { /* and end with state ldstate) search for states and identify them */
        sprintf(token,"_s%d_",i);

        val = seqInputs[ListFinder(seqInputs,token)].position;
      }
      else
      {
        if((i < ldstates + input_linecount - 2) /* one of the normal lines.. */
       && (strposofxcond == -1)) /* make sure we have input at all */
      /* strposofxcond is -1 if there was no input and 'y' is a dummy input*/
        {
          val = seqInputs[ListFinder(seqInputs,
                input_tokenlist[i-ldstates].tok_name)].position;
        }
        else
        {
      val = seqInputs[gndindex].position;
        }
      }
      BX_TRY(BestXDirectRegWrite(handle,seqAddr+MUXOFFSET+4*(bx_int32)i,sizeof(bx_int32),val));
    }

    if (sequencer==BX_SEQ_TRIG)
    {
      BX_TRY_PROGRESS(BestXAnalyzerRunMode(handle,0 /* no soft force */));
    }
    else
    {
      if(mephisto_version == BX_MEPHISTO_MMC1_MMC2)
      {
        /* START OF BYTECOUNTER BUGFIX */
        /* copy all sequencer settings of SEQ0,1 to SEQ 2,3 */
        /* modify the byte counter ce and incr ce to match the original bytecounter ce */
        /* set bytecounter 2,3 to increment mode */
        
        /* copy the mux settings */
        for (i=0;i<7;i++)
        {
          BX_TRY(BestXDirectRegRead(handle,seqAddr+MUXOFFSET+4*(bx_int32)i,sizeof(bx_int32),&val));
          BX_TRY(BestXDirectRegWrite(handle,seqAddr+(BX_PERFSEQ_ADDR_OFFS * 2)+MUXOFFSET+4*(bx_int32)i,sizeof(bx_int32),val));
        }
        
        /* copy the preload value */
        BX_TRY(BestXDirectRegRead(handle,seqAddr+PLOADOFFSET,sizeof(bx_int32),&val));
        BX_TRY(BestXDirectRegWrite(handle,seqAddr+(BX_PERFSEQ_ADDR_OFFS * 2)+PLOADOFFSET,sizeof(bx_int32),val));
        
        
        /* copy the memory content and modfify the  incr ce bit */
        for (i=0;i<k;i++)
        {
          /* set read address */
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_PERF_RAM_PL_REG,sizeof(bx_int16),(bx_int32)i));
          
          /* then assert the strobe once */
          BX_TRY(BestXDirectRegWrite(handle,(BX_REG_PERF_RAM_ADDR_PL_STRB),sizeof(bx_int16),0x0000));
          
          BX_TRY(BestXDirectRegRead(handle,seqAddr,sizeof(bx_int16),&val));
          /* Assumption: incrementor ce  bit position = 7
          byte counter en bit position = 6 */
          /* copy bit 6 to bit 7 */
          mask6 = 0x40;
          mask7 = 0xFFFFFF7F;
          
          val = ((val & mask7 ) | ((val & mask6) << 1));
          /* set write address */
          /* just assert the strobe again */
          BX_TRY(BestXDirectRegWrite(handle,(BX_REG_PERF_RAM_ADDR_PL_STRB),sizeof(bx_int16),0x0000));
          
          
          BX_TRY(BestXDirectRegWrite(handle,seqAddr + (BX_PERFSEQ_ADDR_OFFS * 2),sizeof(bx_int16),val));
        }
        
        
        
        /* set byte counter mode fix to increment by one */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_BLK2_BC_EN_REG,sizeof(bx_int16),0x0));
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_BLK3_BC_EN_REG,sizeof(bx_int16),0x0));
        
        
        /*  END OF BYTECOUNTER BUGFIX */
      }

      BX_TRY_PROGRESS(BestXPerfRunMode(handle,0 /* no soft force */));
    }

    BX_TRY_FAIL(BX_EFW_INTERNAL_RETURN); /* makes sure we run into the catch block */


    } /* BX_TRY_BEGIN */

    /* -------------------------------------------------------------*/
    /* ----------------- freeing the memory again ------------------*/
    /* -------------------------------------------------------------*/

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    { 
      /* switching to progmode successful */
      BX_TRY_PASSED
      {
        /* switching to runmode successful */
      }
      BX_TRY_FAILED
      {
        /* switching to runmode not successful */
        if (sequencer==BX_SEQ_TRIG)
        {
          (void)BestXAnalyzerRunMode(handle,1 /* soft force */);
        }
        else
        {
          (void)BestXPerfRunMode(handle,1 /* soft force */);
        }
      }
    }
    BX_TRY_FAILED
    {
      /* switching to progmode not successful */
    }

    /* now free the allocated memory again */
    FreeSequencerDescriptor(); 


    /* first the allocated memory for the states */
    for(i=0;i<ldstates;i++)
    {
      BestXMemFree((void **)&(state_tokenlist[i].tok_name)); /* ignore return value */
    }
    BestXMemFree((void**)&state_tokenlist);

    /* then the allocated memory of the output token list */
    /* be careful, only the last few allocated outputs are to be deleted */
    /* the first 'used_ldoutputs' outputs are static signals */
    for(i=0;i<(max_ldstates - ldstates);i++) 
    {
      BestXMemFree((void**)&(output_tokenlist[used_ldoutputs+i].tok_name));
    }
    BestXMemFree((void**)&output_tokenlist);

    /* then the allocated memory of the input token list */
    /* no names here */
    BestXMemFree((void**)&input_tokenlist);

    if(BX_TRY_RET == BX_EFW_INTERNAL_RETURN)
    {
      BX_TRY_RET = BX_E_OK;
       
      DBG_OUT(("  No Error occured while programming Sequencer %d \n",sequencer));
      
    }
    else
    {
#if 0 /* chris TBD: Rename BX_EFW_ errors */
      BX_E_ERROR_MSG_SET("Error while programming Sequencer");
      BX_TRY_RET = BX_E_ERROR;
      DBG_OUT(("  **ERROR** while programming Sequencer %d \n",sequencer));
#endif
    }
  } /* CATCH */

  DBG_OUT(("  Finished Programming Sequencer %d \n",sequencer));

  BX_ERRETURN(BX_TRY_RET);
}

